
## ------- Figure 2: Figures & Statistics -------

# ------- Data preparation ---------------------

library(scales)
ste = function(x){
  sd(x)/sqrt(length(x))
}

inputDirectory = '/Users/gfivash/Dropbox/Greg/Work/PhD/Thesis Chapters/Chapter 3 - Sediment hummocks/Data/Growth response/Salicornia Optimal Stopping/Data/'
setwd(inputDirectory)

d = read.csv('Salicornia growth time series.csv')

d$date = as.POSIXct(d$date)
d$pot.ID = factor(d$pot.ID)
d$sediment_shape = factor(d$sediment_shape, levels = c('lowered','flat','raised'))
d$tissue_type = factor(d$tissue_type, levels = c('cotyledon', 'stem', 'green', 'secondary stem'))

cotyledonArea.cm2 = as.vector(tapply(d$area.cm2, list(d$pot.ID, d$tissue, d$date), unique)[,1,])
stemArea.cm2 = as.vector(tapply(d$area.cm2, list(d$pot.ID, d$tissue, d$date), unique)[,2,])
greenArea.cm2 = as.vector(tapply(d$area.cm2, list(d$pot.ID, d$tissue, d$date), unique)[,3,])
secondStemArea.cm2 = as.vector(tapply(d$area.cm2, list(d$pot.ID, d$tissue, d$date), unique)[,4,])

totalArea.cm2 = stemArea.cm2 + greenArea.cm2 + cotyledonArea.cm2 + secondStemArea.cm2
# Area-to-dry mass calibration (See Figure S)
Biomass.g = exp(-4.60303)*(totalArea.cm2^1.15896)
Biomass.mg = Biomass.g * 1000

sedimentShape = factor(rep(levels(d$sediment_shape)[tapply(d$sediment_shape, d$pot.ID, unique)], times = length(unique(d$date))),
                       levels = c('lowered', 'flat','raised'))
pot.ID = rep(unique(d$pot.ID), times = length(unique(d$date)))
date = rep(unique(d$date), each = length(unique(d$pot.ID)))
dayNumber = (as.numeric(date) - as.numeric(date)[1])/(60*60*24)

greenArea.prop = greenArea.cm2/(totalArea.cm2 - cotyledonArea.cm2)
stemArea.prop = stemArea.cm2/(totalArea.cm2 - cotyledonArea.cm2)


# ------- Figures -------

# plotDirectory = '/Users/gfivash/Desktop/'
# setwd(plotDirectory)
# tiff("Figure 4.tif",
#      width = 7, height = 6, 
#      units = 'in', res = 300)

filter = sedimentShape == 'raised'
median = exp(tapply(log(Biomass.mg[filter]), dayNumber[filter], mean))
sd = exp(tapply(log(Biomass.mg[filter]), dayNumber[filter], sd))
length = exp(tapply(log(Biomass.mg[filter]), dayNumber[filter], length))
se = sd/sqrt(length)
se[is.nan(se)] = 0
plot(Biomass.mg[filter] ~ dayNumber[filter], pch = 20, cex = 0.1, ylim = c(1,160), col = 'darkblue', log ='',
     ylab = 'Individual dry biomass (mg)', xlab = 'Days since germination')
individualSeq = as.numeric(unique(pot.ID[filter]))
for(individual in 1:length(individualSeq)){
  individualFilter = pot.ID == individualSeq[individual]
  lines(Biomass.mg[individualFilter] ~ dayNumber[individualFilter], col = alpha('blue',0.1), lwd = 1.5)
}


filter = sedimentShape == 'flat'
individualSeq = as.numeric(unique(pot.ID[filter]))
for(individual in 1:length(individualSeq)){
  individualFilter = pot.ID == individualSeq[individual]
  lines(Biomass.mg[individualFilter] ~ dayNumber[individualFilter], col = alpha('green',0.1), lwd = 1.5)
}
median = exp(tapply(log(Biomass.mg[filter]), dayNumber[filter], mean))
sd = exp(tapply(log(Biomass.mg[filter]), dayNumber[filter], sd))
length = exp(tapply(log(Biomass.mg[filter]), dayNumber[filter], length))
se = sd/sqrt(length)
se[is.nan(se)] = 0
points(Biomass.mg[filter] ~ dayNumber[filter], pch = 20, cex = 0.2, ylim = c(0,100), col = 'darkgreen')



filter = sedimentShape == 'lowered'
individualSeq = as.numeric(unique(pot.ID[filter]))
for(individual in 1:length(individualSeq)){
  individualFilter = pot.ID == individualSeq[individual]
  lines(Biomass.mg[individualFilter] ~ dayNumber[individualFilter], col = alpha('yellow4',0.1), lwd = 1.5)
}
points(Biomass.mg[filter] ~ dayNumber[filter], pch = 20, cex = 0.2, ylim = c(0,100), col = 'khaki')


filter = sedimentShape == 'raised'
median = exp(tapply(log(Biomass.mg[filter]), dayNumber[filter], mean))
sd = exp(tapply(log(Biomass.mg[filter]), dayNumber[filter], sd))
length = exp(tapply(log(Biomass.mg[filter]), dayNumber[filter], length))
se = sd/sqrt(length)
se[is.nan(se)] = 0
points(median ~ unique(dayNumber[filter]), pch = 21, bg = alpha('blue',0.5))
arrows(x0 = unique(dayNumber[filter]), x1 = unique(dayNumber[filter]), y0 = median + sd, y1 = median - sd, 
       length = 0.01, angle = 90, code = 3)


filter = sedimentShape == 'flat'
median = exp(tapply(log(Biomass.mg[filter]), dayNumber[filter], mean))
sd = exp(tapply(log(Biomass.mg[filter]), dayNumber[filter], sd))
length = exp(tapply(log(Biomass.mg[filter]), dayNumber[filter], length))
se = sd/sqrt(length)
se[is.nan(se)] = 0
points(median ~ unique(dayNumber[filter]), pch = 21, bg = alpha('green',0.5))
arrows(x0 = unique(dayNumber[filter]), x1 = unique(dayNumber[filter]), y0 = median + sd, y1 = median - sd, 
       length = 0.01, angle = 90, code = 3)

filter = sedimentShape == 'lowered'
median = exp(tapply(log(Biomass.mg[filter]), dayNumber[filter], mean))
sd = exp(tapply(log(Biomass.mg[filter]), dayNumber[filter], sd))
length = exp(tapply(log(Biomass.mg[filter]), dayNumber[filter], length))
se = sd/sqrt(length)
se[is.nan(se)] = 0
points(median ~ unique(dayNumber[filter]), pch = 21, bg = alpha('yellow',0.5))
arrows(x0 = unique(dayNumber[filter]), x1 = unique(dayNumber[filter]), y0 = median + sd, y1 = median - sd, 
       length = 0.01, angle = 90, code = 3)

legend('topleft', legend = c('Raised','Flat','Lowered'), 
       fill = alpha(c('blue','green','yellow'),0.5), bty = 'n')
# dev.off()


# ------- Statistics -------
colSeq = c('yellow','green','blue')
for(i_sed in 1:length(levels(sedimentShape))){
  filter = sedimentShape == levels(sedimentShape)[i_sed]
  
  y = Biomass.mg[filter][Biomass.mg[filter] != 0]
  x = dayNumber[filter][Biomass.mg[filter] != 0]
  plot(y ~ x, log ='yx', main = levels(sedimentShape)[i_sed],
       pch = 21, bg = alpha(colSeq[i_sed],0.5),
       ylab = 'Dry biomass (mg)', xlab = 'Day number')
  
  
  # Model comparison:
  # Linear growth function
  m0 = lm(y ~ x)
  #Exponential growth function
  m1 = lm(log(y) ~ x)
  #Scaling function
  m2 = lm(log(y) ~ log(x))
  #S-Curve (Gompertz)
  # Test parameters:
  a = 60
  b = 17
  c = 0.03
  m3 = nls(y ~ a*exp(-b*exp(-c*x)), 
           start = list(a = a, b = b, c = c)) 
  AIC(m0,m1,m2,m3)
  
  # Best fit: scaling function
  a = summary(m2)$coefficients[1]
  b = summary(m2)$coefficients[2]
  xv = seq(min(x, na.rm = T),max(x, na.rm = T),0.01)
  yv = exp(a + b*log(xv))
  lines(xv,yv, col = alpha(colSeq[i_sed],0.8), lwd = 2)
  summary(m2)
}

par(mfrow = c(1,3))
sedShape = c()
aSeq = c()
bSeq = c()
individualSeq = c(17:32,1:16,33:48)
for(individual in 1:length(individualSeq)){
  filter = pot.ID == individualSeq[individual] & dayNumber >= 10
  y = Biomass.mg[filter][Biomass.mg[filter] != 0]
  x = dayNumber[filter][Biomass.mg[filter] != 0]
  
  sedShape[individual] = unique(sedimentShape[filter])
  
  #Scaling function
  m2 = lm(log(y) ~ log(x))
  # Best fit: scaling function
  a = summary(m2)$coefficients[1]
  b = summary(m2)$coefficients[2]
  xv = seq(min(x, na.rm = T),max(x, na.rm = T),0.01)
  yv = exp(a + b*log(xv))
  summary(m2)
  
  if(individual == 1){
    plot(y ~ x, log ='yx',
         ylab = 'Dry biomass (mg)', xlab = 'Day number',
         main = unique(levels(sedimentShape)[sedShape[individual]]), 
         pch = 21, bg = alpha(colSeq[sedShape[individual]],0.5), col = alpha('black',0),
         ylim = c(0.001,1000), xlim = c(10,150))
  } else if (sedShape[individual] != sedShape[individual - 1]){
    plot(y ~ x, log ='yx',
         ylab = 'Dry biomass (mg)', xlab = 'Day number',
         main = unique(levels(sedimentShape)[sedShape[individual]]),
         pch = 21, bg = alpha(colSeq[sedShape[individual]],0.5), col = alpha('black',0),
         ylim = c(0.001,1000), xlim = c(10,150))
  } else {
    points(y ~ x,
           pch = 21, bg = alpha(colSeq[sedShape[individual]],0.5), col = alpha('black',0))
  }
  lines(y ~ x,
        pch = 21, bg = alpha(colSeq[sedShape[individual]],0.5))
  lines(xv,yv, col = alpha(colSeq[sedShape[individual]],0.8), lwd = 2)
  
  aSeq[individual] = a
  bSeq[individual] = b
}
par(mfrow = c(1,1))

sedShape = factor(levels(sedimentShape)[sedShape])
sedShape = factor(sedShape,  levels = c('lowered','flat','raised'))
plot(bSeq ~ sedShape, col = alpha(colSeq,0.5), pch= '', ylim = c(1.5, 3.1),
     ylab = 'Rate parameter', xlab = 'Sediment shape')
points(bSeq ~ sedShape, pch = 21, bg = alpha(colSeq[sedShape],0.5))
text(c('A','A','B'), y = c(3,3,3), x = c(1,2,3))

m = aov(bSeq ~ sedShape)
anova(m)
TukeyHSD(m)

p_value = c()
for(i_day in 9:length(unique(dayNumber))){
  filter = dayNumber == unique(dayNumber)[i_day] & Biomass.mg != 0
  m = aov(log(Biomass.mg[filter]) ~ sedimentShape[filter])
  p_value[i_day] = anova(m)[1,5]
}
plot(p_value ~ unique(dayNumber), log = 'y', ylim = c(0.0000001,1), type = 'l',
     ylab = 'p-value', xlab = 'Day number')
legend('topleft', legend = c('Significant','Non-significant'), pch = 20, col = c('red','black'), bty = 'n', cex = 0.8)
points(p_value ~ unique(dayNumber), pch = 20, col = (p_value < 0.05) + 1)
text(y = c(0.1,0.05,0.01,0.005,0.001)*1.4, x = rep(150,times = 5), paste('p =',c(0.1,0.05,0.01,0.005,0.001)), cex = 0.8)
lines(y = c(0.1,0.1), x = c(0, 160), lty = 3)
lines(y = c(0.05,0.05), x = c(0, 160), lty = 3)
lines(y = c(0.01,0.01), x = c(0, 160), lty = 3)
lines(y = c(0.005,0.005), x = c(0, 160), lty = 3)
lines(y = c(0.001,0.001), x = c(0, 160), lty = 3)


p_value[unique(dayNumber) == 56]




filter = dayNumber == unique(dayNumber)[i_day] & Biomass.mg != 0
m = aov(log(Biomass.mg[filter]) ~ sedimentShape[filter])
anova(m)
TukeyHSD(m)


# Reporting average percent comparison values in text:

meanRaised = mean(bSeq[sedShape == 'raised'])
meanNotRaised = mean(bSeq[sedShape != 'raised'])
percentDiff = (meanRaised/meanNotRaised - 1) * 100

deviationRaised.perc = abs(bSeq[sedShape == 'raised'] - meanRaised)/meanRaised * 100
deviationNotRaised.perc = abs(bSeq[sedShape != 'raised'] - meanNotRaised)/meanNotRaised * 100
meanDeviation.perc = mean(c(deviationNotRaised.perc/sqrt(32), deviationRaised.perc/sqrt(16)))

paste(round(percentDiff,2), '', round(meanDeviation.perc,2))

# ---------------
meanR = mean(log(Biomass.mg[dayNumber == max(dayNumber) & sedimentShape == 'raised']))
meanNotR = mean(log(Biomass.mg[dayNumber == max(dayNumber) & sedimentShape != 'raised']))
percentDiff = (exp(meanR)/exp(meanNotR) - 1) * 100

deviationRaised.perc = as.numeric(abs(exp(log(Biomass.mg[dayNumber == max(dayNumber) & sedimentShape == 'raised']) - meanR))/exp(meanR) * 100)/sqrt(16)
deviationNotRaised.perc = as.numeric(abs(exp(log(Biomass.mg[dayNumber == max(dayNumber) & sedimentShape != 'raised']) - meanNotR))/exp(meanNotR) * 100)/sqrt(32)
meanDeviation.perc = exp(mean(log(c(deviationRaised.perc, deviationNotRaised.perc))))

paste(round(percentDiff,2), '', round(meanDeviation.perc,2))




meanR = mean(log(Biomass.mg[dayNumber == max(dayNumber) & sedimentShape == 'raised']))
meanNotR = mean(log(Biomass.mg[dayNumber == max(dayNumber) & sedimentShape != 'raised']))
percentDiff = (exp(meanR)/exp(meanNotR) - 1) * 100

deviationRaised.perc = as.numeric(abs(exp(log(Biomass.mg[dayNumber == max(dayNumber) & sedimentShape == 'raised']) - meanR))/exp(meanR) * 100)/sqrt(16)
deviationNotRaised.perc = as.numeric(abs(exp(log(Biomass.mg[dayNumber == max(dayNumber) & sedimentShape != 'raised']) - meanNotR))/exp(meanNotR) * 100)/sqrt(32)
meanDeviation.perc = exp(mean(log(c(deviationRaised.perc, deviationNotRaised.perc))))

paste(round(percentDiff,2), '', round(meanDeviation.perc,2))

